<?php
error_reporting(E_ALL ^ E_NOTICE);

header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Content-Type: application/json; charset=utf-8"); 

include_once '../conexion.php'; 
include_once '../clases/TokenValidator.php';

try {
    // ✅ Validar Token antes de cualquier consulta
    $usuarioId = TokenValidator::validar($conn);

    // ✅ Obtener y validar datos de la solicitud
    $bodyRequest = file_get_contents("php://input");
    $datos = json_decode($bodyRequest, true);

    if (!is_array($datos) || !isset($datos[0]) || !is_numeric($datos[0])) {
        error_response("El parámetro empresa_id es obligatorio y debe ser numérico.");
    }

    $empresa_id = (int) $datos[0];
    $rows = [];

    // ✅ Preparar llamada a función PostgreSQL
    $sql = "SELECT * FROM listar_inconsistencia_doc(:empresa_id)";
    $stmt = $conn->prepare($sql);
    $stmt->bindParam(':empresa_id', $empresa_id, PDO::PARAM_INT);
    $stmt->execute();

    // ✅ Recoger resultados
    if ($stmt->rowCount() > 0) {
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $rows[] = $row;
        }
    }

    // ✅ Devolver resultados
    print_json($rows);

} catch (PDOException $e) {
    error_response("Error en la consulta SQL: " . $e->getMessage());
} catch (Exception $e) {
    error_response("Error general: " . $e->getMessage());
} finally {
    $conn = null; // ✅ Cerrar conexión explícitamente
}

// =======================
// ✅ Funciones auxiliares
// =======================
function print_json($data) {
    echo json_encode($data, JSON_UNESCAPED_UNICODE);

    switch (json_last_error()) {
        case JSON_ERROR_NONE:
            break;
        case JSON_ERROR_DEPTH:
            error_response('Máxima profundidad de pila excedida');
            break;
        case JSON_ERROR_STATE_MISMATCH:
            error_response('Desbordamiento o desajuste de modos');
            break;
        case JSON_ERROR_CTRL_CHAR:
            error_response('Caracter de control inesperado');
            break;
        case JSON_ERROR_SYNTAX:
            error_response('Error de sintaxis, JSON mal formado');
            break;
        case JSON_ERROR_UTF8:
            error_response('Caracteres UTF-8 mal formados');
            break;
        default:
            error_response('Error JSON desconocido');
            break;
    }
}

function error_response($message) {
    http_response_code(500);
    echo json_encode(['error' => $message], JSON_UNESCAPED_UNICODE);
    exit();
}
?>