<?php
error_reporting(E_ALL ^ E_NOTICE);

header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Content-Type: application/json; charset=utf-8");

// Incluir la conexión a PostgreSQL
include_once '../conexion.php';
include_once '../clases/TokenValidator.php';

// Llama la validación antes de cualquier consulta
$usuarioId = TokenValidator::validar($conn);

try {
    // Obtener el cuerpo de la solicitud
    $bodyRequest = file_get_contents("php://input");
    $datos = json_decode($bodyRequest, true);

    // Validar que los parámetros sean correctos
    if (!isset($datos[0], $datos[1], $datos[2], $datos[3], $datos[4], $datos[5], $datos[6])) {
        throw new Exception("Parámetros inválidos: Se requieren siete valores.");
    }

    $sql = "SELECT * FROM listar_ventas_n(:param1, :param2, :param3, :param4, :param5, :param6, :param7)";
    $stmt = $conn->prepare($sql);

    // Asignar parámetros
    $stmt->bindValue(':param1', $datos[0], PDO::PARAM_INT);
    $stmt->bindValue(':param2', $datos[1], PDO::PARAM_INT);
    $stmt->bindValue(':param3', $datos[2], PDO::PARAM_STR);
    $stmt->bindValue(':param4', $datos[3], PDO::PARAM_STR);
    $stmt->bindValue(':param5', $datos[4], PDO::PARAM_STR);
    $stmt->bindValue(':param6', $datos[5], PDO::PARAM_INT);
    $stmt->bindValue(':param7', $datos[6], PDO::PARAM_INT);

    // Ejecutar la consulta
    $stmt->execute();

    // Obtener los resultados
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Verificar si hay datos
    if (!$rows) {
        error_log("⚠️ No se encontraron ventas con esos filtros.");
    }

    // Enviar JSON
    print_json($rows);

} catch (Exception $e) {
    // Manejo de errores
    error_log("❌ Error en Listar_Ventas: " . $e->getMessage());
    error_response("Error: " . $e->getMessage());
} finally {
    // Liberar memoria y cerrar la conexión
    unset($stmt);
    $conn = null;
}

/**
 * Imprime datos en formato JSON y maneja errores de codificación
 */
function print_json($data) {
    echo json_encode($data, JSON_UNESCAPED_UNICODE);

    switch (json_last_error()) {
        case JSON_ERROR_NONE:
            break;
        case JSON_ERROR_DEPTH:
            error_response('Maximum stack depth exceeded');
            break;
        case JSON_ERROR_STATE_MISMATCH:
            error_response('Underflow or the modes mismatch');
            break;
        case JSON_ERROR_CTRL_CHAR:
            error_response('Unexpected control character found');
            break;
        case JSON_ERROR_SYNTAX:
            error_response('Syntax error, malformed JSON');
            break;
        case JSON_ERROR_UTF8:
            error_response('Malformed UTF-8 characters, possibly incorrectly encoded');
            break;
        default:
            error_response('Unknown error');
            break;
    }
}

/**
 * Devuelve un mensaje de error en formato JSON
 */
function error_response($message) {
    http_response_code(500);
    echo json_encode(['error' => $message], JSON_UNESCAPED_UNICODE);
    exit();
}
?>